<?php
/**
 * Functions
 *
 * @package     AutomatorWP\Mautic\Functions
 * @since       1.0.0
 */
// Exit if accessed directly
if (!defined('ABSPATH')) exit;

/**
 * Get the request parameters
 *
 * @since 1.0.0
 *
 * @param string $platform
 *
 * @return array|false
 */
function automatorwp_mautic_get_request_parameters( ) {

    $auth = get_option( 'automatorwp_mautic_auth' );
    $disable_sslverify = (bool) automatorwp_mautic_get_option( 'disable_ssl', false );
    
    if( ! is_array( $auth ) ) {
        return false;
    }

    $params = array(
        'user-agent'  => 'AutomatorWP; ' . home_url(),
        'timeout'     => 120,
        'httpversion' => '1.1',
        'headers'     => array(
            'Authorization' => 'Bearer ' . $auth['access_token'],
            'Content-Type'  => 'application/json',
            'Accept'        => 'application/json'
        )
    );

    // Disable SSL verify
    if ( $disable_sslverify )
        $params['sslverify'] = false;

    return $params;
}

/**
 * Get the request parameters
 *
 * @since 1.0.0
 *
 * @param string $platform
 *
 * @return string|false|WP_Error
 */
function automatorwp_mautic_refresh_token( ) {

    $client_id = automatorwp_mautic_get_option( 'client_id', '' );
    $client_secret = automatorwp_mautic_get_option( 'client_secret', '' );
    $mautic_url = automatorwp_mautic_get_option( 'url', '' );
    $disable_sslverify = (bool) automatorwp_mautic_get_option( 'disable_ssl', false );

    if( empty( $client_id ) || empty( $client_secret ) ) {
        return false;
    }

    $auth = get_option( 'automatorwp_mautic_auth', false );

    if( ! is_array( $auth ) ) {
        return false;
    }

    $params = array(
        'headers' => array(
            'Content-Type'  => 'application/x-www-form-urlencoded; charset=utf-8',
            'Accept'        => 'application/json',
        ),
        'body'  => array(            
            'client_id'     => $client_id,
            'client_secret' => $client_secret,
            'grant_type'    => 'refresh_token',
            'refresh_token' => $auth['refresh_token'],
        ),
    );

    // Disable SSL verify
    if ( $disable_sslverify )
        $params['sslverify'] = false;
            
    $response = wp_remote_post( $mautic_url . '/oauth/v2/token', $params );

    if( is_wp_error( $response ) ) {
        return false;
    }

    $response_code = wp_remote_retrieve_response_code( $response );

    if ( $response_code !== 200 ) {
        return false;
    }

    $body = json_decode( wp_remote_retrieve_body( $response ) );

    $auth = array(
        'access_token'  => $body->access_token,
        'refresh_token' => $body->refresh_token,
        'token_type'    => $body->token_type,
        'expires_in'    => $body->expires_in,
        'scope'         => $body->scope,
    );

    // Update the access and refresh tokens
    update_option( 'automatorwp_mautic_auth', $auth );

    return $body->access_token;

}

/**
 * Filters the HTTP API response immediately before the response is returned.
 *
 * @since 1.0.0
 *
 * @param array  $response    HTTP response.
 * @param array  $parsed_args HTTP request arguments.
 * @param string $url         The request URL.
 *
 * @return array
 */
function automatorwp_mautic_maybe_refresh_token( $response, $args, $url ) {

    $mautic_url = automatorwp_mautic_get_option( 'url', '' );

    // Ensure to only run this check to on Google API request
    if( strpos( $url, $mautic_url ) !== false ) {
        
        $code = wp_remote_retrieve_response_code( $response );
        
        if( $code === 401 ) {

            $access_token = automatorwp_mautic_refresh_token( );

            // Send again the request if token gets refreshed successfully
            if( $access_token ) {

                $args['headers']['Authorization'] = 'Bearer ' . $access_token;

                $response = wp_remote_request( $url, $args );

            }

        }

    }

    return $response;

}
add_filter( 'http_response', 'automatorwp_mautic_maybe_refresh_token', 10, 3 );

/**
 * Get contact ID
 *
 * @since 1.0.0
 *
 * @param int    $email         Email
 * 
 */
function automatorwp_mautic_get_contact_id ( $email ) {

    $params = automatorwp_mautic_get_request_parameters(  );

    // Bail if the authorization has not been setup from settings
    if( $params === false ) {
        return false;
    }

    $contact_id = '';

    $mautic_url = automatorwp_mautic_get_option( 'url', '' );

    $response = wp_remote_get( $mautic_url . '/api/contacts?search=' . urlencode( $email ), $params );

    $response = json_decode( wp_remote_retrieve_body( $response ), true  );
    
    foreach ( $response['contacts'] as $contact ) {
        $contact_id = isset( $contact['id'] ) ? $contact['id'] : '';
    }

    return $contact_id;
    
}

/**
 * Add contact to Mautic
 *
 * @since 1.0.0
 * 
 * @param array     $contact     The contact data
 */
function automatorwp_mautic_add_contact( $contact ) {

    $params = automatorwp_mautic_get_request_parameters(  );

    // Bail if the authorization has not been setup from settings
    if( $params === false ) {
        return false;
    }

    $params['body'] = json_encode( $contact );

    $mautic_url = automatorwp_mautic_get_option( 'url', '' );

    $response = wp_remote_post( $mautic_url . '/api/contacts/new', $params );
   
    $code = wp_remote_retrieve_response_code( $response );
    
    return $code;

}

/**
 * Remove contact from Mautic
 *
 * @since 1.0.0
 * 
 * @param int     $contact_id     The contact ID
 * 
 * @return int
 */
function automatorwp_mautic_remove_contact( $contact_id ) {

    $params = automatorwp_mautic_get_request_parameters(  );

    // Bail if the authorization has not been setup from settings
    if( $params === false ) {
        return false;
    }

    $params['method'] = 'DELETE';

    $mautic_url = automatorwp_mautic_get_option( 'url', '' );

    $response = wp_remote_request( $mautic_url . '/api/contacts/' . $contact_id . '/delete', $params );
   
    $code = wp_remote_retrieve_response_code( $response );
    
    return $code;

}

/**
 * Get segments from Mautic
 *
 * @since 1.0.0
 * 
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_mautic_options_cb_segment( $field ) {

    // Setup vars
    $value = $field->escaped_value;
    $none_value = 'any';
    $none_label = __( 'any segment', 'automatorwp-pro' );
    $options = automatorwp_options_cb_none_option( $field, $none_value, $none_label );
        
    if( ! empty( $value ) ) {
        if( ! is_array( $value ) ) {
            $value = array( $value );
        }
    
        foreach( $value as $segment_id ) {
    
            // Skip option none
            if( $segment_id === $none_value ) {
                continue;
            }
                
            $options[$segment_id] = automatorwp_mautic_get_segment_name( $segment_id );
        }
    }
    
    return $options;

}

/**
 * Get segments from Mautic
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_mautic_get_segments( ) {

    $segments = array();

    $transient = get_transient( 'automatorwp_mautic_segments' );

    if ( $transient !== false ) {
        return $transient;
    }

    $params = automatorwp_mautic_get_request_parameters(  );

    // Bail if the authorization has not been setup from settings
    if( $params === false ) {
        return false;
    }

    $mautic_url = automatorwp_mautic_get_option( 'url', '' );

    $response = wp_remote_get( $mautic_url . '/api/segments', $params );

    $response = json_decode( wp_remote_retrieve_body( $response ), true  );
    
    if ( empty ( $response ) )
        return $segments;

    foreach ( $response['lists'] as $segment ) {
        $segments[] = array(
            'id' => $segment['id'],
            'name' => $segment['name'],
        );

    }

    if( count( $segments ) ) {
        // Set a transient for 10 mins with the segments
        set_transient( 'automatorwp_mautic_segments', $segments, 10 * 60 );
    }

    return $segments;

}

/**
 * Get segment name
 *
 * @since 1.0.0
 *
 * @param int    $segment_id         ID segment
 * 
 */
function automatorwp_mautic_get_segment_name( $segment_id ) {

    // Empty title if no ID provided
    if( absint( $segment_id ) === 0 ) {
        return '';
    }

    $params = automatorwp_mautic_get_request_parameters(  );

    // Bail if the authorization has not been setup from settings
    if( $params === false ) {
        return false;
    }

    $mautic_url = automatorwp_mautic_get_option( 'url', '' );

    $response = wp_remote_get( $mautic_url . '/api/segments/' . $segment_id, $params );

    $response = json_decode( wp_remote_retrieve_body( $response ), true  );

    return $response['list']['name'];

}

/**
 * Add contact to segment
 *
 * @since 1.0.0
 * @param int     $segment_id     The segment ID 
 * @param int     $contact_id     The contact ID
 * 
 * @return int
 */
function automatorwp_mautic_add_contact_segment( $segment_id, $contact_id ) {

    $params = automatorwp_mautic_get_request_parameters(  );

    // Bail if the authorization has not been setup from settings
    if( $params === false ) {
        return false;
    }

    $mautic_url = automatorwp_mautic_get_option( 'url', '' );

    $response = wp_remote_post( $mautic_url . '/api/segments/' . $segment_id . '/contact/' . $contact_id . '/add', $params );
     
    $code = wp_remote_retrieve_response_code( $response );
    
    return $code;

}

/**
 * Remove contact from segment
 *
 * @since 1.0.0
 * @param int     $segment_id     The segment ID 
 * @param int     $contact_id     The contact ID
 * 
 * @return int
 */
function automatorwp_mautic_remove_contact_segment( $segment_id, $contact_id ) {

    $params = automatorwp_mautic_get_request_parameters(  );

    // Bail if the authorization has not been setup from settings
    if( $params === false ) {
        return false;
    }

    $mautic_url = automatorwp_mautic_get_option( 'url', '' );

    $response = wp_remote_post( $mautic_url . '/api/segments/' . $segment_id . '/contact/' . $contact_id . '/remove', $params );
     
    $code = wp_remote_retrieve_response_code( $response );
    
    return $code;

}

/**
 * Get companies from Mautic
 *
 * @since 1.0.0
 * 
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_mautic_options_cb_company( $field ) {

    // Setup vars
    $value = $field->escaped_value;
    $none_value = 'any';
    $none_label = __( 'any company', 'automatorwp-pro' );
    $options = automatorwp_options_cb_none_option( $field, $none_value, $none_label );
        
    if( ! empty( $value ) ) {
        if( ! is_array( $value ) ) {
            $value = array( $value );
        }
    
        foreach( $value as $company_id ) {
    
            // Skip option none
            if( $company_id === $none_value ) {
                continue;
            }
                
            $options[$company_id] = automatorwp_mautic_get_company_name( $company_id );
        }
    }
    
    return $options;

}

/**
 * Get companies from Mautic
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_mautic_get_companies( ) {

    $companies = array();

    $transient = get_transient( 'automatorwp_mautic_companies' );

    if ( $transient !== false ) {
        return $transient;
    }

    $params = automatorwp_mautic_get_request_parameters(  );

    // Bail if the authorization has not been setup from settings
    if( $params === false ) {
        return false;
    }

    $mautic_url = automatorwp_mautic_get_option( 'url', '' );

    $response = wp_remote_get( $mautic_url . '/api/companies', $params );

    $response = json_decode( wp_remote_retrieve_body( $response ), true  );

    if ( empty ( $response ) )
        return $companies;

    foreach ( $response['companies'] as $company ) {
    
        $companies[] = array(
            'id' => $company['fields']['all']['id'],
            'name' => $company['fields']['all']['companyname'],
        );

    }

    if( count( $companies ) ) {
        // Set a transient for 10 mins with the companies
        set_transient( 'automatorwp_mautic_companies', $companies, 10 * 60 );
    }

    return $companies;

}

/**
 * Get company name
 *
 * @since 1.0.0
 *
 * @param int    $company_id         ID company
 * 
 */
function automatorwp_mautic_get_company_name( $company_id ) {

    // Empty title if no ID provided
    if( absint( $company_id ) === 0 ) {
        return '';
    }

    $params = automatorwp_mautic_get_request_parameters(  );

    // Bail if the authorization has not been setup from settings
    if( $params === false ) {
        return false;
    }

    $mautic_url = automatorwp_mautic_get_option( 'url', '' );

    $response = wp_remote_get( $mautic_url . '/api/companies/' . $company_id, $params );

    $response = json_decode( wp_remote_retrieve_body( $response ), true  );

    return $response['company']['fields']['all']['companyname'];

}

/**
 * Add contact to company
 *
 * @since 1.0.0
 * 
 * @param int     $company_id     The company ID
 * @param int     $contact_id     The contact ID
 * 
 * @return int
 */
function automatorwp_mautic_add_contact_company( $company_id, $contact_id ) {

    $params = automatorwp_mautic_get_request_parameters(  );

    // Bail if the authorization has not been setup from settings
    if( $params === false ) {
        return false;
    }

    $mautic_url = automatorwp_mautic_get_option( 'url', '' );

    $response = wp_remote_post( $mautic_url . '/api/companies/' . $company_id . '/contact/' . $contact_id . '/add', $params );
     
    $code = wp_remote_retrieve_response_code( $response );
    
    return $code;

}

/**
 * Remove contact from company
 *
 * @since 1.0.0
 * 
 * @param int     $company_id     The company ID
 * @param int     $contact_id     The contact ID
 * 
 * @return int
 */
function automatorwp_mautic_remove_contact_company( $company_id, $contact_id ) {

    $params = automatorwp_mautic_get_request_parameters(  );

    // Bail if the authorization has not been setup from settings
    if( $params === false ) {
        return false;
    }

    $mautic_url = automatorwp_mautic_get_option( 'url', '' );

    $response = wp_remote_post( $mautic_url . '/api/companies/' . $company_id . '/contact/' . $contact_id . '/remove', $params );
     
    $code = wp_remote_retrieve_response_code( $response );
    
    return $code;

}

/**
 * Get a list of contact’s companies the contact belongs to.
 *
 * @since 1.0.0
 *
 * @param int    $contact_id         contact ID
 * 
 * @return array
 */
function automatorwp_mautic_get_contact_companies( $contact_id ) {

    $contact_companies = array();

    $params = automatorwp_mautic_get_request_parameters(  );

    // Bail if the authorization has not been setup from settings
    if( $params === false ) {
        return false;
    }

    $mautic_url = automatorwp_mautic_get_option( 'url', '' );

    $response = wp_remote_get( $mautic_url . '/api/contacts/' . $contact_id . '/companies', $params );

    $response = json_decode( wp_remote_retrieve_body( $response ), true  );
   
    foreach ( $response['companies'] as $company ) {
        $contact_companies[] = absint( $company['id'] );
    }

    return $contact_companies;

}

/**
 * Get a list of contact segments the contact is a member of.
 *
 * @since 1.0.0
 *
 * @param int    $contact_id         contact ID
 * 
 * @return array
 */
function automatorwp_mautic_get_contact_segments( $contact_id ) {

    $contact_segments = array();

    $params = automatorwp_mautic_get_request_parameters(  );

    // Bail if the authorization has not been setup from settings
    if( $params === false ) {
        return false;
    }

    $mautic_url = automatorwp_mautic_get_option( 'url', '' );

    $response = wp_remote_get( $mautic_url . '/api/contacts/' . $contact_id . '/segments', $params );

    $response = json_decode( wp_remote_retrieve_body( $response ), true  );
   
    foreach ( $response['lists'] as $segment ) {
        $contact_segments[] = absint( $segment['id'] );
    }

    return $contact_segments;

}

/**
 * Get contact fields from Mautic
 *
 * @since 1.0.0
 * 
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_mautic_options_cb_contact_field( $field ) {

    // Setup vars
    $value = $field->escaped_value;
    $none_value = 'any';
    $none_label = __( 'any field', 'automatorwp-pro' );
    $options = automatorwp_options_cb_none_option( $field, $none_value, $none_label );
        
    if( ! empty( $value ) ) {
        if( ! is_array( $value ) ) {
            $value = array( $value );
        }
    
        foreach( $value as $field_id ) {
    
            // Skip option none
            if( $field_id === $none_value ) {
                continue;
            }
                
            $options[$field_id] = automatorwp_mautic_get_contact_field_name( $field_id );
        }
    }
    
    return $options;

}

/**
 * Get a list of contact fields.
 *
 * @since 1.0.0
 * 
 * @return array
 */
function automatorwp_mautic_get_contact_fields( ) {

    $contact_fields = array();

    $transient = get_transient( 'automatorwp_mautic_fields' );

    if ( $transient !== false ) {
        return $transient;
    }

    $params = automatorwp_mautic_get_request_parameters(  );

    // Bail if the authorization has not been setup from settings
    if( $params === false ) {
        return false;
    }

    $mautic_url = automatorwp_mautic_get_option( 'url', '' );

    $response = wp_remote_get( $mautic_url . '/api/fields/contact', $params );

    $response = json_decode( wp_remote_retrieve_body( $response ), true  );
   
    foreach ( $response['fields'] as $field ) {
        $contact_fields[] = array(
            'id' => $field['id'],
            'name' => $field['label'],
        );

    }

    if( count( $contact_fields ) ) {
        // Set a transient for 10 mins with the contact fields
        set_transient( 'automatorwp_mautic_fields', $contact_fields, 10 * 60 );
    }

    return $contact_fields;

}

/**
 * Get contact field name
 *
 * @since 1.0.0
 *
 * @param int    $field_id         ID Contact Field
 * 
 * @return string
 */
function automatorwp_mautic_get_contact_field_name( $field_id ) {

    $params = automatorwp_mautic_get_request_parameters(  );

    // Bail if the authorization has not been setup from settings
    if( $params === false ) {
        return false;
    }

    $mautic_url = automatorwp_mautic_get_option( 'url', '' );

    $response = wp_remote_get( $mautic_url . '/api/fields/contact/' . $field_id, $params );

    $response = json_decode( wp_remote_retrieve_body( $response ), true  );

    return $response['field']['label'];

}

/**
 * Get a list of contact fields with alias.
 *
 * @since 1.0.0
 * 
 * @return array
 */
function automatorwp_mautic_get_contact_fields_alias( ) {

    $contact_fields = array();

    $transient = get_transient( 'automatorwp_mautic_fields_alias' );

    if ( $transient !== false ) {
        return $transient;
    }

    $params = automatorwp_mautic_get_request_parameters(  );

    // Bail if the authorization has not been setup from settings
    if( $params === false ) {
        return false;
    }

    $mautic_url = automatorwp_mautic_get_option( 'url', '' );

    $response = wp_remote_get( $mautic_url . '/api/fields/contact', $params );

    $response = json_decode( wp_remote_retrieve_body( $response ), true  );
   
    foreach ( $response['fields'] as $field ) {
        $contact_fields[$field['id']] = $field['alias'];
    }

    if( count( $contact_fields ) ) {
        // Set a transient for 10 mins with the contact fields alias
        set_transient( 'automatorwp_mautic_fields_alias', $contact_fields, 10 * 60 );
    }

    return $contact_fields;

}

/**
 * Get contact field alias.
 *
 * @since 1.0.0
 * 
 * @param int    $field_id         ID Contact Field
 * 
 * @return array
 */
function automatorwp_mautic_get_contact_field_alias( $field_id ) {

    // Empty title if no ID provided
    if( absint( $field_id ) === 0 ) {
        return '';
    }
    
    // Get ID and Alias
    $contact_fields = automatorwp_mautic_get_contact_fields_alias();

    return $contact_fields[$field_id];

}

